// Load and manage metadata for agorum objects
/* global sc */

function moduleFunction(config) {
  // Initialize $$ with provided config or default
  let $$ = config || require('./debug')({ debug: 'cache', path: module.id });

  let objects = require('common/objects').sc($$.environment.context || sc);
  let metadata = require('common/metadata');
  let beans = require('common/beans');

  /**
   * Loads metadata for the specified object.
   * 
   * @param {string} obj - The object to load metadata for.
   * @param {...*} var_args - Additional arguments.
   * @returns {Object} - The loaded metadata.
   */
  function load(obj, var_args) {
    arguments['0'] = objects.find(obj);
    if (arguments.length === 1) {
      [].push.call(arguments, /^((?!content$))/);
    }
    let m = metadata();
    return m.load.apply(m, arguments).data();
  }

  /**
   * Saves metadata for the specified object.
   * 
   * @param {string} obj - The object to save metadata for.
   * @param {Object} data - The metadata to save.
   * @returns {Object} - The saved metadata.
   */
  function save(obj, data) {
    return metadata(data).save(objects.find(obj));
  }

  /**
   * Load inherited metadata from given agorum object.
   *
   * @param {*} obj - agorum object spec
   * @returns {Object} metadata objects with inheritance array
   */
  function inherited(obj) {
    let result = {},
      raw = beans.get(objects.find(obj), '~~');

    for (var key in raw) {
      let values = raw[key];
      result[key] = values.length > 1 ? values : values[0];
    }
    return result;
  }

  /**
   * Load all (including inherited) metadata.
   *
   * @param {*} obj - agorum object reference
   * @returns {Object} - All metadata for the object
   */
  function all(obj) {
    return metadata.all(objects.find(obj));
  }

  // Return the module functions
  return {
    load: load,
    save: save,
    inherited: inherited,
    all: all
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
